﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Diagnostics;
using System.Xml.Linq;
using System.Text.RegularExpressions;
using System.IO;
using Tar = ICSharpCode.SharpZipLib.Tar;


namespace BSS.Backup
{
    static class Utils
    {
        static public void ExecuteActions(IEnumerable<XElement> actions)
        {
            foreach(XElement element in actions)
            {
                XAttribute attr = element.Attribute("exec");
                if(attr == null || string.IsNullOrEmpty(attr.Value))
                    continue;
                ProcessStartInfo psi = new ProcessStartInfo(attr.Value);
                attr = element.Attribute("dir");
                if(attr != null && !string.IsNullOrEmpty(attr.Value))
                    psi.WorkingDirectory = attr.Value;
                attr = element.Attribute("args");
                if(attr != null && !string.IsNullOrEmpty(attr.Value))
                    psi.Arguments = attr.Value;
                using(Process process = Process.Start(psi))
                {
                    int runTimeout = int.MaxValue;
                    attr = element.Attribute("timeout");
                    if(attr != null && !string.IsNullOrEmpty(attr.Value))
                        runTimeout = int.Parse(attr.Value) * 1000;
                    process.WaitForExit(runTimeout);
                }
            }
        }

        static public bool MatchFilePattern(string fileName, string pattern)
        {
            if(string.IsNullOrEmpty(fileName))
                throw new ArgumentException("invalid file name");
            if(string.IsNullOrEmpty(pattern))
                throw new ArgumentException("invalid pattern");

            // check first and last chars for speed up comparission
            char c = pattern[0];
            if(c != '*' && c != '?' && char.ToLowerInvariant(fileName[0]) != char.ToLowerInvariant(c))
                return false;

            // Ok, now we should make the full comparison
            pattern = pattern.ToLowerInvariant();
            fileName = fileName.ToLowerInvariant();

            short plen = (short) pattern.Length;
            short[] gaps = new short[plen];
            short cp = (short) (fileName.Length - 1);
            short ppos = (short) (plen - 1);
            while(true)
            {
                c = pattern[ppos];
                if(c == '*')
                {
                    gaps[ppos] = cp;
                    if(--ppos < 0)
                    {
                        return true;
                    }
                    continue;
                }
                if(cp >= 0 && (c == '?' || c == fileName[cp]))
                {
                    gaps[ppos] = cp--;
                    if(--ppos >= 0)
                    {
                        continue;
                    }
                    if(cp < 0)
                    {
                        return true;
                    }
                }
                do
                {
                    while(++ppos < plen && pattern[ppos] != '*')
                        ;
                    if(ppos == plen)
                        return false;
                    while(ppos + 1 < plen && pattern[ppos + 1] == '*')
                        ++ppos;
                    cp = gaps[ppos];
                } while(--cp < 0);
                gaps[ppos--] = cp;
            }
        }
    }

    struct BackupSource
    {
        public readonly string Path;
        public readonly byte DepthLevel;

        public BackupSource(string path, byte depthLevel)
        {
            this.Path = path;
            this.DepthLevel = depthLevel;
        }
    }

    class Program
    {
        static TraceSource trace;

        static void Main(string[] args)
        {
            trace = new TraceSource("BssBackup", SourceLevels.Information);

            if(args.Length > 2)
            {
                Console.Error.WriteLine("BssBackup - simple FTP backup\nUsage BssBackup.exe [<ConfigFile> [<BackupLevel>]]");
                System.Environment.Exit(2);
                return;
            }
            trace.TraceInformation("Start");
            bool exactLevelMatch = false;
            byte backupLevel = 3;
            if(args.Length > 1)
            {
                if(args[1].StartsWith("="))
                {
                    backupLevel = byte.Parse(args[1].Substring(1));
                    exactLevelMatch = true;
                }
                else
                {
                    backupLevel = byte.Parse(args[1]);
                }
            }
            else
            {
                DateTime nextDay = DateTime.UtcNow.AddDays(1);
                if(nextDay.DayOfYear == 1)
                {
                    backupLevel = 0;
                }
                else if(nextDay.Day == 1)
                {
                    backupLevel = 1;
                }
                else if(nextDay.DayOfWeek == DayOfWeek.Monday)
                {
                    backupLevel = 2;
                }
            }
            trace.TraceEvent(TraceEventType.Verbose, 0, "Set min backup level to {0}.", backupLevel);
            // load config
            XDocument config = XDocument.Load(args.Length > 0 ? args[0]
                : Path.GetFileNameWithoutExtension(Environment.GetCommandLineArgs()[0]) + ".xml");
            // start up backup system
            Utils.ExecuteActions(config.Root.Elements("startup"));
            try
            {
                // build groups of backup sets
                var queryBackupNames = from bs in config.Root.Elements("backup_set")
                                       let bsName = bs.Attribute("name").Value
                                       from bsl in bs.Attribute("level").Value.Split(new char[] { ',', ' ' }, StringSplitOptions.RemoveEmptyEntries)
                                       let bsLevel = byte.Parse(bsl)
                                       where bsLevel == backupLevel || (!exactLevelMatch && bsLevel > backupLevel)
                                       group new { BackupSet = bs, Level = bsLevel} by bsName;

                foreach(var bsg in queryBackupNames)
                {
                    string backupName = bsg.Key;
                    byte realBackupLevel = bsg.Min(bs => bs.Level);
                    trace.TraceInformation("Backup definition for {0} with level {1} was found.", backupName, realBackupLevel);
                    var catalogConfig = (from c in config.Root.Elements("catalog")
                                         where c.Attribute("name").Value == backupName
                                         select c).First();
                    if(catalogConfig == null)
                    {
                        trace.TraceEvent(TraceEventType.Error, 0, "Catalog definition for backup {0} not found. Skip backup.", backupName);
                        continue;
                    }
                    int retentionQuantity = -1;
                    foreach(XElement retentionElement in catalogConfig.Elements("retention"))
                    {
                        XAttribute attr = retentionElement.Attribute("level");
                        if(attr == null || byte.Parse(attr.Value) != realBackupLevel)
                            continue;
                        attr = retentionElement.Attribute("quantity");
                        if(attr != null)
                            retentionQuantity = Math.Max(retentionQuantity, int.Parse(attr.Value) - 1);
                    }
                    if(retentionQuantity < 0)
                    {
                        retentionQuantity = int.MaxValue;
                    }
                    using(Catalog catalog = new Catalog(catalogConfig))
                    {
                        catalog.Open();
                        trace.TraceInformation("Catalog {0} is open.", backupName);
                        var catalogEntries = new List<CatalogEntry>(catalog.EnumEntries());
                        DateTime lastDateDifferential = (from cf in catalogEntries
                                                         where cf.Level < realBackupLevel
                                                         select cf.StartDate).DefaultIfEmpty(DateTime.MinValue).Max();
                        DateTime lastDateIncremental = (from cf in catalogEntries
                                                        where cf.Level <= realBackupLevel
                                                        select cf.StartDate).DefaultIfEmpty(DateTime.MinValue).Max();
                        catalogEntries = catalogEntries.Where(cf => cf.Level == realBackupLevel).OrderByDescending(cf => cf.StartDate).Skip(retentionQuantity).ToList();
                        // Start catalog backup //
                        DateTime backupStart = DateTime.UtcNow;
                        int fileCount = 0;
                        Tar.TarArchive archive;
                        CatalogEntry newEntry = catalog.Append(backupStart, realBackupLevel, out archive);
                        using(archive)
                        {
                            foreach(var bc in bsg.Where(bs => bs.Level == realBackupLevel).Select(bs => bs.BackupSet))
                            {
                                // determine type and set trailTimestamp. Differential is default.
                                var attr = bc.Attribute("type");
                                DateTime trailTimestamp = lastDateDifferential;
                                if(attr != null)
                                {
                                    switch(attr.Value.ToLower())
                                    {
                                        case "incremental":
                                            trailTimestamp = lastDateIncremental;
                                            break;
                                        case "full":
                                            trailTimestamp = DateTime.MinValue;
                                            break;
                                    }
                                }
                                fileCount += ProcessBackupSet(archive, bc, trailTimestamp);
                            }
                            archive.Close();
                        }
                        if(fileCount == 0)
                        {
                            catalog.Delete(newEntry);
                        }
                        else
                        {
                            // Delete old backups // 
                            catalogEntries.ForEach(catalog.Delete);
                        }
                        catalog.Close();
                    }
                }
            }
            finally
            {
                // shut down backup system
                Utils.ExecuteActions(config.Root.Elements("shutdown"));
            }
            trace.TraceInformation("Finish");
        }

        static int ProcessBackupSet(Tar.TarArchive archive, XElement bc, DateTime trailTimestamp)
        {
            int fileCount = 0;
            string backupName = bc.Attribute("name").Value;
            trace.TraceInformation("Prepare backup {0}", backupName);
            Utils.ExecuteActions(bc.Elements("startup"));
            var excludeFiles = (from el in bc.Elements("exclude")
                                let a = el.Attribute("file")
                                where a != null && !string.IsNullOrEmpty(a.Value)
                                select a.Value).ToList();

            var includeFiles = (from el in bc.Elements("include")
                                let a = el.Attribute("file")
                                where a != null && !string.IsNullOrEmpty(a.Value)
                                select a.Value).ToList();

            var excludePaths = (from el in bc.Elements("exclude")
                                let a = el.Attribute("path")
                                where a != null && !string.IsNullOrEmpty(a.Value)
                                select a.Value).ToList();

            var includePaths = (from el in bc.Elements("include")
                                let a = el.Attribute("path")
                                where a != null && !string.IsNullOrEmpty(a.Value)
                                select a.Value).ToList();

            var excludePatterns = (from el in bc.Elements("exclude")
                                   let a = el.Attribute("pattern")
                                   where a != null && !string.IsNullOrEmpty(a.Value)
                                   select new Regex(a.Value,
                                       RegexOptions.CultureInvariant |
                                       RegexOptions.IgnoreCase |
                                       RegexOptions.Compiled)).ToList();

            var includePatterns = (from el in bc.Elements("include")
                                   let a = el.Attribute("pattern")
                                   where a != null && !string.IsNullOrEmpty(a.Value)
                                   select new Regex(a.Value,
                                       RegexOptions.CultureInvariant |
                                       RegexOptions.IgnoreCase |
                                       RegexOptions.Compiled)).ToList();

            Stack<BackupSource> processingStack = new Stack<BackupSource>();
            foreach(var backupBase in from el in bc.Elements("base")
                                      let ap = el.Attribute("path")
                                      where ap != null && !string.IsNullOrEmpty(ap.Value)
                                      select new
                                      {
                                          Element = el,
                                          Path = ap.Value
                                      })
            {
                XAttribute attr = backupBase.Element.Attribute("level");
                processingStack.Push(new BackupSource(backupBase.Path, attr == null ? byte.MaxValue : byte.Parse(attr.Value)));
                attr = backupBase.Element.Attribute("name");
                archive.RootPath = backupBase.Path.Replace(Path.DirectorySeparatorChar, '/');
                archive.PathPrefix = (attr == null ? null : attr.Value);
                while(processingStack.Count > 0)
                {
                    BackupSource backupSource = processingStack.Pop();
                    var directoryInfo = new DirectoryInfo(backupSource.Path);
                    foreach(var fileInfo in directoryInfo.GetFiles())
                    {
                        if(fileInfo.LastWriteTimeUtc <= trailTimestamp)
                            continue;
                        bool include = excludeFiles.Count > 0 || excludePatterns.Count > 0
                            || (includeFiles.Count == 0 && includePatterns.Count == 0);
                        string shortName = fileInfo.Name;
                        string fullName = fileInfo.FullName;
                        foreach(var p in excludeFiles)
                        {
                            if(Utils.MatchFilePattern(shortName, p))
                            {
                                include = false;
                                break;
                            }
                        }
                        if(!include)
                        {
                            foreach(var p in includeFiles)
                            {
                                if(Utils.MatchFilePattern(shortName, p))
                                {
                                    include = true;
                                    break;
                                }
                            }
                        }
                        if(include)
                        {
                            foreach(var p in excludePatterns)
                            {
                                if(p.IsMatch(fullName))
                                {
                                    include = false;
                                    break;
                                }
                            }
                        }
                        if(!include)
                        {
                            foreach(var p in includePatterns)
                            {
                                if(p.IsMatch(fullName))
                                {
                                    include = true;
                                    break;
                                }
                            }
                        }
                        if(include)
                        {
                            trace.TraceInformation("Add file {0}", fullName);
                            Tar.TarEntry entry = Tar.TarEntry.CreateEntryFromFile(fullName);
                            fullName = fullName.Replace(Path.DirectorySeparatorChar, '/');
                            entry.Name = fullName;
                            archive.WriteEntry(entry, false);
                            ++fileCount;
                        }
                        else
                        {
                            trace.TraceInformation("Skip file {0}", fullName);
                        }
                    }
                    if(backupSource.DepthLevel > 0)
                    {
                        foreach(var dirInfo in directoryInfo.GetDirectories())
                        {
                            bool include = excludePaths.Count > 0 || includePaths.Count == 0;
                            string fullName = dirInfo.FullName;
                            foreach(var p in excludePaths)
                            {
                                if(Utils.MatchFilePattern(fullName, p))
                                {
                                    include = false;
                                    break;
                                }
                            }
                            if(!include)
                            {
                                foreach(var p in includePaths)
                                {
                                    if(Utils.MatchFilePattern(fullName, p))
                                    {
                                        include = true;
                                        break;
                                    }
                                }
                            }
                            string patternName = fullName + Path.DirectorySeparatorChar;
                            if(include)
                            {
                                foreach(var p in excludePatterns)
                                {
                                    if(p.IsMatch(patternName))
                                    {
                                        include = false;
                                        break;
                                    }
                                }
                            }
                            if(!include)
                            {
                                foreach(var p in includePatterns)
                                {
                                    if(p.IsMatch(patternName))
                                    {
                                        include = true;
                                        break;
                                    }
                                }
                            }
                            if(include)
                            {
                                processingStack.Push(new BackupSource(fullName, (byte) (backupSource.DepthLevel - 1)));
                            }
                            else
                            {
                                trace.TraceInformation("Skip subdirectory{0}", fullName);
                            }
                        }
                    }
                }
            }
            trace.TraceInformation("Shutdown backup {0}", backupName);
            Utils.ExecuteActions(bc.Elements("shutdown"));
            return fileCount;
        }
    }
}
