﻿using System;
using System.Collections.Generic;
using System.Net;
using System.IO;
using System.Threading;

namespace BSS.Backup
{
    class FtpOutputStream : Stream
    {
        const int MAX_QUEUE_LENGTH = 1 << 22;

        Uri location;
        AutoResetEvent queueFree;
        ManualResetEvent enqueueAllowed;
        ManualResetEvent queueEmpty;

        LinkedList<byte[]> writeQueue;
        long storedLegth;
        long queueLength;
        FtpWebRequest ftpRequest;
        Stream ftpWriteStream;

        public FtpOutputStream(Uri location)
        {
            this.location = location;
            this.writeQueue = new LinkedList<byte[]>();
            this.queueFree = new AutoResetEvent(true);
            this.enqueueAllowed = new ManualResetEvent(true);
            this.queueEmpty = new ManualResetEvent(true);
        }

        public override bool CanSeek
        {
            get
            {
                return false;
            }
        }

        public override bool CanRead
        {
            get
            {
                return false;
            }
        }

        public override bool CanWrite
        {
            get
            {
                return true;
            }
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            if(count <= 0)
                return;
            WaitHandle.WaitAll(new WaitHandle[] { this.enqueueAllowed, this.queueFree });
            try
            {
                byte[] datapacket = new byte[count];
                Array.Copy(buffer, offset, datapacket, 0, count);
                this.writeQueue.AddLast(datapacket);
                if(this.queueLength == 0)
                {
                    this.queueEmpty.Reset();
                    ThreadPool.QueueUserWorkItem(new WaitCallback(this.BackgroundWriter));
                }
                this.queueLength += count;
                if(this.queueLength >= MAX_QUEUE_LENGTH)
                {
                    this.enqueueAllowed.Reset();
                }
                if(this.ftpRequest == null)
                {
                    this.ftpRequest = (FtpWebRequest) FtpWebRequest.Create(this.location);
                    this.ftpRequest.KeepAlive = false;
                    this.ftpRequest.Method = this.storedLegth > 0 ? WebRequestMethods.Ftp.AppendFile : WebRequestMethods.Ftp.UploadFile;
                    this.ftpRequest.Timeout = Timeout.Infinite;
                    this.ftpRequest.UseBinary = true;
                    this.ftpWriteStream = this.ftpRequest.GetRequestStream();
                }
            }
            finally
            {
                this.queueFree.Set();
            }
        }

        void BackgroundWriter(object state)
        {
            byte[] datapacket;
            int dataLength = 0;
            while(true)
            {
                this.queueFree.WaitOne();
                try
                {
                    this.storedLegth += dataLength;
                    if(this.queueLength >= MAX_QUEUE_LENGTH && this.queueLength - dataLength < MAX_QUEUE_LENGTH)
                    {
                        this.enqueueAllowed.Set();
                    }
                    this.queueLength -= dataLength;
                    if(this.writeQueue.Count == 0)
                    {
                        this.queueEmpty.Set();
                        return;
                    }
                    datapacket = this.writeQueue.First.Value;
                    this.writeQueue.RemoveFirst();
                }
                finally
                {
                    this.queueFree.Set();
                }
                dataLength = datapacket.Length;
                this.ftpWriteStream.Write(datapacket, 0, dataLength);
            }
        }

        public override void Flush()
        {
            this.queueEmpty.WaitOne();
            System.Diagnostics.Debug.Assert(this.queueLength == 0);
        }

        public override void Close()
        {
            this.enqueueAllowed.Reset();    // Stop queue //
            if(WaitHandle.WaitAll(new WaitHandle[] { this.queueFree, this.queueEmpty }))
            {
                if(this.ftpRequest != null)
                {
                    this.ftpWriteStream.Close();
                    this.ftpWriteStream = null;
                    FtpWebResponse response = (FtpWebResponse) this.ftpRequest.GetResponse();
                    Stream respStream = response.GetResponseStream();
                    respStream.Close();
                    response.Close();
                    this.ftpRequest = null;
                }
                this.writeQueue = null;
                this.queueEmpty.Close();
                this.queueFree.Close();
                this.enqueueAllowed.Close();
            }
        }

        protected override void Dispose(bool disposing)
        {
            if(disposing)
            {
                if(this.writeQueue != null)
                {
                    this.Close();
                }
            }
            base.Dispose(disposing);
        }

        public override long Length
        {
            get
            {
                return this.storedLegth + this.queueLength;
            }
        }

        public override long Position
        {
            get
            {
                return this.storedLegth + this.queueLength;
            }
            set
            {
                throw new NotSupportedException();
            }
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            throw new NotSupportedException();
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            throw new NotSupportedException();
        }

        public override void SetLength(long value)
        {
            throw new NotSupportedException();
        }
    }

}
