﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Xml.Linq;
using System.Net;
using System.Text.RegularExpressions;
using System.IO;
using System.Threading;
using Tar = ICSharpCode.SharpZipLib.Tar;
using BZip = ICSharpCode.SharpZipLib.BZip2;
using GZip = ICSharpCode.SharpZipLib.GZip;
using ZipCore = ICSharpCode.SharpZipLib.Core;

namespace BSS.Backup
{
    public static class TextReaderExtender
    {
        public static IEnumerable<string> SplitLines(this TextReader reader)
        {
            string line;
            while((line = reader.ReadLine()) != null)
                yield return line;
        }
    }

    class CatalogEntry
    {
        public readonly DateTime StartDate;
        public readonly byte Level;
        public readonly string FileName;

        public CatalogEntry(DateTime startDate, byte level, string fileName)
        {
            this.StartDate = startDate;
            this.Level = level;
            this.FileName = fileName;
        }
    }

    class Catalog : IDisposable
    {
        XElement config;
        bool open;

        private Uri GetLocation()
        {
            return new Uri(this.config.Element("location").Attribute("path").Value);
        }

        private string GetName()
        {
            return this.config.Attribute("name").Value;
        }

        public Catalog(XElement config)
        {
            this.config = config;
        }
        public void Open()
        {
            if(!this.open)
            {
                Utils.ExecuteActions(this.config.Elements("startup"));
                this.open = true;
            }
        }
        public void Close()
        {
            if(this.open)
            {
                this.open = false;
                Utils.ExecuteActions(this.config.Elements("shutdown"));
            }
        }
        public IEnumerable<CatalogEntry> EnumEntries()
        {
            IEnumerable<string> files = null;
            bool wasOpen = this.IsOpen;
            this.Open();
            try
            {
                Uri location = this.GetLocation();
                if(location.IsFile || location.IsUnc)
                {
                    files = (from fi in (new DirectoryInfo(location.LocalPath)).GetFiles()
                             select fi.Name).ToList();
                }
                else if(location.Scheme == Uri.UriSchemeFtp)
                {
                    FtpWebRequest req = (FtpWebRequest) WebRequest.Create(location);
                    req.Method = WebRequestMethods.Ftp.ListDirectory;
                    req.KeepAlive = false;
                    FtpWebResponse res = (FtpWebResponse) req.GetResponse();
                    try
                    {
                        Stream respStream = res.GetResponseStream();
                        if(res.StatusCode == FtpStatusCode.OpeningData || res.StatusCode == FtpStatusCode.DataAlreadyOpen)
                        {
                            using(StreamReader reader = new StreamReader(respStream))
                            {
                                files = reader.SplitLines().ToList();
                            }
                        }
                        else
                        {
                            respStream.Close();
                        }
                    }
                    finally
                    {
                        res.Close();
                    }
                }
                else // if (uri.Scheme == Uri.UriSchemeHttp || uri.Scheme == Uri.UriSchemeHttps)
                {
                    // WebDav hier
                    throw new NotImplementedException("Unsupported protocol");
                }
            }
            finally
            {
                if(!wasOpen)
                    this.Close();
            }
            if(files != null)
            {
                Regex regexParser = new Regex("(?<c>[^-]+)-(?<y>\\d{2})(?<m>\\d{2})(?<d>\\d{2})(?<h>\\d{2})(?<mi>\\d{2})(?<s>\\d{2})-(?<l>\\d+)\\.tar(\\.gz|\\.bzip2|\\.bz2)?");
                string catalogName = this.GetName();
                foreach(var fileName in files)
                {
                    Match m = regexParser.Match(fileName);
                    if(!m.Success || m.Groups["c"].Value != catalogName)
                        continue;

                    yield return new CatalogEntry(
                        new DateTime(
                            2000 + int.Parse(m.Groups["y"].Value),
                            int.Parse(m.Groups["m"].Value),
                            int.Parse(m.Groups["d"].Value),
                            int.Parse(m.Groups["h"].Value),
                            int.Parse(m.Groups["mi"].Value),
                            int.Parse(m.Groups["s"].Value),
                            DateTimeKind.Utc),
                        byte.Parse(m.Groups["l"].Value),
                        fileName
                        );
                }
            }
        }
        public CatalogEntry Append(DateTime crtTime, byte level, out Tar.TarArchive archive)
        {
            if(!this.open)
            {
                throw new ApplicationException("Catalog should be open");
            }
            string fileName = string.Format("{0}-{1:yyMMddHHmmss}-{2}.tar", this.GetName(), crtTime.ToUniversalTime(), level);
            string compressionMethod = null;
            byte compressionLevel = 3;
            XElement compressionConfig = this.config.Element("compression");
            if(compressionConfig != null)
            {
                XAttribute attr = compressionConfig.Attribute("method");
                if(attr != null)
                {
                    compressionMethod = attr.Value.ToLowerInvariant();
                    if(compressionMethod == "gzip")
                    {
                        fileName += ".gz";
                    }
                    else if(compressionMethod == "bzip2")
                    {
                        fileName += ".bz2";
                    }
                    else
                    {
                        compressionMethod = null;
                    }
                }
                attr = compressionConfig.Attribute("level");
                if(attr != null)
                {
                    compressionLevel = byte.Parse(attr.Value);
                }
            }
            Stream outStream;
            Uri catalogLocation = this.GetLocation();
            if(catalogLocation.IsFile || catalogLocation.IsUnc)
            {
                outStream = new FileStream(Path.Combine(catalogLocation.AbsolutePath, fileName), FileMode.Create, FileAccess.Write, FileShare.None);
            }
            else if(catalogLocation.Scheme == Uri.UriSchemeFtp)
            {
                outStream = new FtpOutputStream(new Uri(catalogLocation, fileName));
            }
            else
            {
                // WebDav hier
                throw new NotImplementedException("Unsupported protocol");
            }
            if(compressionMethod != null)
            {
                switch(compressionMethod)
                {
                    case "bzip2":
                        BZip.BZip2OutputStream bzipOutStream = new BZip.BZip2OutputStream(new BufferedStream(outStream));
                        bzipOutStream.IsStreamOwner = true;
                        outStream = bzipOutStream;
                        break;

                    case "gzip":
                        GZip.GZipOutputStream gzipOutStream = new GZip.GZipOutputStream(new BufferedStream(outStream));
                        gzipOutStream.IsStreamOwner = true;
                        gzipOutStream.SetLevel(compressionLevel);
                        outStream = gzipOutStream;
                        break;
                }
            }
            archive = Tar.TarArchive.CreateOutputTarArchive(outStream);
            archive.AsciiTranslate = false;
            return new CatalogEntry(crtTime, level, fileName);
        }
        public void Delete(CatalogEntry catalogFile)
        {
            bool wasOpen = this.IsOpen;
            this.Open();
            try
            {
                Uri location = this.GetLocation();
                if(location.IsFile || location.IsUnc)
                {
                    File.Delete(Path.Combine(location.LocalPath, catalogFile.FileName));
                }
                else if(location.Scheme == Uri.UriSchemeFtp)
                {
                    FtpWebRequest req = (FtpWebRequest) WebRequest.Create(new Uri(location, catalogFile.FileName));
                    req.Method = WebRequestMethods.Ftp.DeleteFile;
                    req.KeepAlive = false;
                    FtpWebResponse res = (FtpWebResponse) req.GetResponse();
                    try
                    {
                        Stream respStream = res.GetResponseStream();
                        respStream.Close();
                        if(res.StatusCode != FtpStatusCode.FileActionOK)
                        {
                            // TODO: Was dann?
                        }
                    }
                    finally
                    {
                        res.Close();
                    }
                }
                else // if (uri.Scheme == Uri.UriSchemeHttp || uri.Scheme == Uri.UriSchemeHttps)
                {
                    // WebDav hier
                    throw new NotImplementedException("Unsupported protocol");
                }
            }
            finally
            {
                if(!wasOpen)
                    this.Close();
            }
        }

        public bool IsOpen
        {
            get
            {
                return this.open;
            }
        }

        protected virtual void Dispose(bool disposing)
        {
            if(this.config != null)
            {
                this.Close();
                this.config = null;
            }
        }

        #region IDisposable Member

        public void Dispose()
        {
            this.Dispose(true);
        }

        #endregion
    }
}
